"use client"

import { useEffect, useState } from "react"
import { useRouter } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Badge } from "@/components/ui/badge"
import {
  Users,
  FileText,
  Heart,
  AlertTriangle,
  Handshake,
  GraduationCap,
  LogOut,
  TrendingUp,
  Clock,
  CheckCircle,
} from "lucide-react"

export default function DashboardPage() {
  const [isAuthenticated, setIsAuthenticated] = useState(false)
  const [userEmail, setUserEmail] = useState("")
  const router = useRouter()

  useEffect(() => {
    const authStatus = localStorage.getItem("isAuthenticated")
    const email = localStorage.getItem("userEmail")

    if (authStatus === "true" && email) {
      setIsAuthenticated(true)
      setUserEmail(email)
    } else {
      router.push("/login")
    }
  }, [router])

  const handleLogout = () => {
    localStorage.removeItem("isAuthenticated")
    localStorage.removeItem("userEmail")
    router.push("/login")
  }

  if (!isAuthenticated) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading dashboard...</p>
        </div>
      </div>
    )
  }

  const stats = [
    {
      title: "Total Volunteers",
      value: "45",
      icon: Users,
      color: "text-blue-600",
      bgColor: "bg-blue-100",
    },
    {
      title: "Program Applications",
      value: "156",
      icon: FileText,
      color: "text-green-600",
      bgColor: "bg-green-100",
    },
    {
      title: "Parent Support Requests",
      value: "28",
      icon: Heart,
      color: "text-purple-600",
      bgColor: "bg-purple-100",
    },
    {
      title: "Incident Reports",
      value: "12",
      icon: AlertTriangle,
      color: "text-red-600",
      bgColor: "bg-red-100",
    },
    {
      title: "Partner Inquiries",
      value: "23",
      icon: Handshake,
      color: "text-orange-600",
      bgColor: "bg-orange-100",
    },
    {
      title: "Active Programs",
      value: "7",
      icon: GraduationCap,
      color: "text-indigo-600",
      bgColor: "bg-indigo-100",
    },
  ]

  const recentActivity = [
    {
      type: "volunteer",
      message: "New volunteer application from Sarah Johnson",
      time: "2 hours ago",
      status: "pending",
    },
    {
      type: "incident",
      message: "Incident report submitted for Pelican Park area",
      time: "4 hours ago",
      status: "urgent",
    },
    {
      type: "parent",
      message: "Parent support request for family counseling",
      time: "6 hours ago",
      status: "in-progress",
    },
    {
      type: "program",
      message: "Skills workshop application completed",
      time: "1 day ago",
      status: "completed",
    },
    {
      type: "partner",
      message: "Partnership inquiry from local NGO",
      time: "2 days ago",
      status: "pending",
    },
  ]

  const volunteers = [
    { name: "John Smith", email: "john@email.com", skills: "Teaching, Mentoring", status: "Active" },
    { name: "Mary Johnson", email: "mary@email.com", skills: "Counseling, Support", status: "Pending" },
    { name: "David Wilson", email: "david@email.com", skills: "IT, Digital Literacy", status: "Active" },
    { name: "Lisa Brown", email: "lisa@email.com", skills: "Healthcare, Wellness", status: "Inactive" },
  ]

  const incidents = [
    { id: "INC-001", type: "Safety Concern", location: "Pelican Park", priority: "High", status: "Open" },
    { id: "INC-002", type: "Child Welfare", location: "Mitchells Plain", priority: "Critical", status: "In Progress" },
    { id: "INC-003", type: "Community Issue", location: "Khayelitsha", priority: "Medium", status: "Resolved" },
  ]

  const parentSupport = [
    { name: "Anonymous", issue: "Teen behavioral issues", urgency: "High", status: "Assigned" },
    { name: "Jane Doe", issue: "Family counseling needed", urgency: "Medium", status: "Pending" },
    { name: "Anonymous", issue: "Child safety concerns", urgency: "Critical", status: "In Progress" },
  ]

  const programs = [
    { name: "Vigulate", enrolled: 45, capacity: 60, completion: 75 },
    { name: "VAZO Link", enrolled: 32, capacity: 40, completion: 80 },
    { name: "Skills Workshops", enrolled: 28, capacity: 35, completion: 85 },
    { name: "Mentorship Circles", enrolled: 22, capacity: 30, completion: 73 },
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="bg-white shadow">
        <div className="container mx-auto px-4 py-4">
          <div className="flex justify-between items-center">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">VAZO Foundation Dashboard</h1>
              <p className="text-gray-600">Welcome back, {userEmail}</p>
            </div>
            <Button onClick={handleLogout} variant="outline">
              <LogOut className="mr-2 h-4 w-4" />
              Logout
            </Button>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-4 py-8">
        {/* Statistics Overview */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
          {stats.map((stat, index) => (
            <Card key={index}>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">{stat.title}</CardTitle>
                <div className={`p-2 rounded-full ${stat.bgColor}`}>
                  <stat.icon className={`h-4 w-4 ${stat.color}`} />
                </div>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stat.value}</div>
                <div className="flex items-center text-xs text-muted-foreground">
                  <TrendingUp className="mr-1 h-3 w-3" />
                  +12% from last month
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Detailed Tabs */}
        <Tabs defaultValue="activity" className="space-y-4">
          <TabsList>
            <TabsTrigger value="activity">Recent Activity</TabsTrigger>
            <TabsTrigger value="volunteers">Volunteers</TabsTrigger>
            <TabsTrigger value="incidents">Incidents</TabsTrigger>
            <TabsTrigger value="parent-support">Parent Support</TabsTrigger>
            <TabsTrigger value="programs">Programs</TabsTrigger>
          </TabsList>

          <TabsContent value="activity" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Recent Activity</CardTitle>
                <CardDescription>Latest updates from all systems</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {recentActivity.map((activity, index) => (
                    <div key={index} className="flex items-center space-x-4 p-4 border rounded-lg">
                      <div className="flex-shrink-0">
                        <Clock className="h-5 w-5 text-gray-400" />
                      </div>
                      <div className="flex-1">
                        <p className="text-sm font-medium">{activity.message}</p>
                        <p className="text-xs text-gray-500">{activity.time}</p>
                      </div>
                      <Badge
                        variant={
                          activity.status === "urgent"
                            ? "destructive"
                            : activity.status === "completed"
                              ? "default"
                              : "secondary"
                        }
                      >
                        {activity.status}
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="volunteers" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Volunteer Applications</CardTitle>
                <CardDescription>Manage volunteer registrations and status</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {volunteers.map((volunteer, index) => (
                    <div key={index} className="flex items-center justify-between p-4 border rounded-lg">
                      <div>
                        <h4 className="font-medium">{volunteer.name}</h4>
                        <p className="text-sm text-gray-600">{volunteer.email}</p>
                        <p className="text-xs text-gray-500">Skills: {volunteer.skills}</p>
                      </div>
                      <Badge
                        variant={
                          volunteer.status === "Active"
                            ? "default"
                            : volunteer.status === "Pending"
                              ? "secondary"
                              : "outline"
                        }
                      >
                        {volunteer.status}
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="incidents" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Incident Reports</CardTitle>
                <CardDescription>Track and manage community incident reports</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {incidents.map((incident, index) => (
                    <div key={index} className="flex items-center justify-between p-4 border rounded-lg">
                      <div>
                        <h4 className="font-medium">{incident.id}</h4>
                        <p className="text-sm text-gray-600">{incident.type}</p>
                        <p className="text-xs text-gray-500">Location: {incident.location}</p>
                      </div>
                      <div className="flex space-x-2">
                        <Badge
                          variant={
                            incident.priority === "Critical"
                              ? "destructive"
                              : incident.priority === "High"
                                ? "destructive"
                                : "secondary"
                          }
                        >
                          {incident.priority}
                        </Badge>
                        <Badge variant={incident.status === "Resolved" ? "default" : "secondary"}>
                          {incident.status}
                        </Badge>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="parent-support" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Parent Support Requests</CardTitle>
                <CardDescription>Manage family support and counseling requests</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {parentSupport.map((request, index) => (
                    <div key={index} className="flex items-center justify-between p-4 border rounded-lg">
                      <div>
                        <h4 className="font-medium">{request.name}</h4>
                        <p className="text-sm text-gray-600">{request.issue}</p>
                      </div>
                      <div className="flex space-x-2">
                        <Badge
                          variant={
                            request.urgency === "Critical"
                              ? "destructive"
                              : request.urgency === "High"
                                ? "destructive"
                                : "secondary"
                          }
                        >
                          {request.urgency}
                        </Badge>
                        <Badge variant="secondary">{request.status}</Badge>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="programs" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Program Enrollment</CardTitle>
                <CardDescription>Track program participation and completion rates</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {programs.map((program, index) => (
                    <div key={index} className="p-4 border rounded-lg">
                      <div className="flex items-center justify-between mb-2">
                        <h4 className="font-medium">{program.name}</h4>
                        <Badge variant="outline">{program.completion}% Complete</Badge>
                      </div>
                      <div className="flex items-center justify-between text-sm text-gray-600">
                        <span>
                          Enrolled: {program.enrolled}/{program.capacity}
                        </span>
                        <div className="flex items-center">
                          <CheckCircle className="mr-1 h-3 w-3" />
                          Active
                        </div>
                      </div>
                      <div className="mt-2 w-full bg-gray-200 rounded-full h-2">
                        <div
                          className="bg-blue-600 h-2 rounded-full"
                          style={{ width: `${(program.enrolled / program.capacity) * 100}%` }}
                        ></div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  )
}
