import type React from "react"
import type { Metadata } from "next"
import { Inter } from "next/font/google"
import "./globals.css"
import { MainNav } from "@/components/main-nav"
import Link from "next/link"
import { Phone, Mail, MapPin, Facebook, Instagram, Linkedin } from "lucide-react"

const inter = Inter({ subsets: ["latin"] })

export const metadata: Metadata = {
  title: "VAZO Foundation NPC - Building Child-Safe Communities",
  description:
    "Empowering youth and building child-safe communities in Pelican Park and surrounding areas through skills development, mentorship, and community safety programs.",
  keywords:
    "youth development, child protection, community safety, skills training, mentorship, Pelican Park, Cape Town",
  authors: [{ name: "VAZO Foundation NPC" }],
  creator: "VAZO Foundation NPC",
  publisher: "VAZO Foundation NPC",
  formatDetection: {
    email: false,
    address: false,
    telephone: false,
  },
  metadataBase: new URL("https://vazofoundation.org"),
  alternates: {
    canonical: "/",
  },
  openGraph: {
    title: "VAZO Foundation NPC - Building Child-Safe Communities",
    description:
      "Empowering youth and building child-safe communities through comprehensive programs and community safety initiatives.",
    url: "https://vazofoundation.org",
    siteName: "VAZO Foundation NPC",
    images: [
      {
        url: "/vazo-foundation-logo.png",
        width: 1200,
        height: 630,
        alt: "VAZO Foundation NPC Logo",
      },
    ],
    locale: "en_ZA",
    type: "website",
  },
  twitter: {
    card: "summary_large_image",
    title: "VAZO Foundation NPC - Building Child-Safe Communities",
    description:
      "Empowering youth and building child-safe communities through comprehensive programs and community safety initiatives.",
    images: ["/vazo-foundation-logo.png"],
  },
  robots: {
    index: true,
    follow: true,
    googleBot: {
      index: true,
      follow: true,
      "max-video-preview": -1,
      "max-image-preview": "large",
      "max-snippet": -1,
    },
  },
  verification: {
    google: "your-google-verification-code",
  },
    generator: 'v0.app'
}

const jsonLd = {
  "@context": "https://schema.org",
  "@type": "NonProfitOrganization",
  name: "VAZO Foundation NPC",
  alternateName: "VAZO Foundation",
  url: "https://vazofoundation.org",
  logo: "https://vazofoundation.org/vazo-foundation-logo.png",
  description:
    "Building child-safe communities through youth empowerment, skills development, and community safety programs in Pelican Park and surrounding areas.",
  address: {
    "@type": "PostalAddress",
    streetAddress: "Pelican Park",
    addressLocality: "Cape Town",
    addressRegion: "Western Cape",
    addressCountry: "ZA",
  },
  contactPoint: {
    "@type": "ContactPoint",
    telephone: "+27-87-822-2686",
    contactType: "customer service",
    availableLanguage: ["English", "Afrikaans", "Xhosa"],
  },
  sameAs: [
    "https://www.facebook.com/vazofoundation",
    "https://www.instagram.com/vazofoundation",
    "https://www.linkedin.com/company/vazo-foundation",
  ],
  foundingDate: "2023",
  areaServed: {
    "@type": "Place",
    name: "Pelican Park and surrounding areas, Cape Town, South Africa",
  },
  knowsAbout: [
    "Youth Development",
    "Child Protection",
    "Community Safety",
    "Skills Training",
    "Mentorship Programs",
    "Job Readiness",
    "Digital Literacy",
  ],
  memberOf: {
    "@type": "Organization",
    name: "South African Non-Profit Organizations",
  },
}

export default function RootLayout({
  children,
}: {
  children: React.ReactNode
}) {
  return (
    <html lang="en">
      <head>
        <script type="application/ld+json" dangerouslySetInnerHTML={{ __html: JSON.stringify(jsonLd) }} />
      </head>
      <body className={inter.className}>
        <div className="min-h-screen flex flex-col">
          <header className="sticky top-0 z-50 w-full border-b bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
            <div className="container flex h-14 items-center">
              <MainNav />
            </div>
          </header>

          <main className="flex-1">{children}</main>

          <footer className="bg-gray-900 text-white py-12">
            <div className="container mx-auto px-4">
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
                {/* Contact Information */}
                <div>
                  <h3 className="text-lg font-semibold mb-4 text-vazoOrange">Contact Us</h3>
                  <div className="space-y-3">
                    <div className="flex items-center space-x-2">
                      <Phone className="h-4 w-4 text-gray-300" />
                      <span className="text-gray-300">0878222686</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Mail className="h-4 w-4 text-gray-300" />
                      <span className="text-gray-300">info@vazofoundation.org</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <MapPin className="h-4 w-4 text-gray-300" />
                      <span className="text-gray-300">Pelican Park, Cape Town</span>
                    </div>
                  </div>
                </div>

                {/* Quick Links */}
                <div>
                  <h3 className="text-lg font-semibold mb-4 text-vazoOrange">Quick Links</h3>
                  <ul className="space-y-2">
                    <li>
                      <Link href="/about/our-story" className="text-gray-300 hover:text-white transition-colors">
                        About Us
                      </Link>
                    </li>
                    <li>
                      <Link href="/programs" className="text-gray-300 hover:text-white transition-colors">
                        Programs
                      </Link>
                    </li>
                    <li>
                      <Link href="/get-involved/volunteer" className="text-gray-300 hover:text-white transition-colors">
                        Volunteer
                      </Link>
                    </li>
                    <li>
                      <Link href="/get-involved/donate" className="text-gray-300 hover:text-white transition-colors">
                        Donate
                      </Link>
                    </li>
                    <li>
                      <Link href="/parent-support" className="text-gray-300 hover:text-white transition-colors">
                        Parent Support
                      </Link>
                    </li>
                  </ul>
                </div>

                {/* Programs */}
                <div>
                  <h3 className="text-lg font-semibold mb-4 text-vazoOrange">Programs</h3>
                  <ul className="space-y-2">
                    <li>
                      <Link
                        href="/programs/skills-workshops"
                        className="text-gray-300 hover:text-white transition-colors"
                      >
                        Skills Workshops
                      </Link>
                    </li>
                    <li>
                      <Link
                        href="/programs/mentorship-circles"
                        className="text-gray-300 hover:text-white transition-colors"
                      >
                        Mentorship Circles
                      </Link>
                    </li>
                    <li>
                      <Link
                        href="/programs/job-readiness-coaching"
                        className="text-gray-300 hover:text-white transition-colors"
                      >
                        Job Readiness
                      </Link>
                    </li>
                    <li>
                      <Link href="/programs/vigilant" className="text-gray-300 hover:text-white transition-colors">
                        Vigilant (In Development)
                      </Link>
                    </li>
                    <li>
                      <Link href="/programs/vazo-link" className="text-gray-300 hover:text-white transition-colors">
                        Vazo Link (Under Construction)
                      </Link>
                    </li>
                  </ul>
                </div>

                {/* Legal & Policies */}
                <div>
                  <h3 className="text-lg font-semibold mb-4 text-vazoOrange">Legal & Policies</h3>
                  <ul className="space-y-2">
                    <li>
                      <Link href="/privacy-policy" className="text-gray-300 hover:text-white transition-colors">
                        Privacy Policy
                      </Link>
                    </li>
                    <li>
                      <Link href="/data-protection" className="text-gray-300 hover:text-white transition-colors">
                        Data Protection
                      </Link>
                    </li>
                    <li>
                      <Link href="/whistleblowing-policy" className="text-gray-300 hover:text-white transition-colors">
                        Whistleblowing Policy
                      </Link>
                    </li>
                    <li>
                      <Link href="/about/child-protection" className="text-gray-300 hover:text-white transition-colors">
                        Child Protection
                      </Link>
                    </li>
                    <li>
                      <Link href="/report-incident" className="text-gray-300 hover:text-white transition-colors">
                        Report Incident
                      </Link>
                    </li>
                  </ul>
                </div>
              </div>

              {/* Social Media & Copyright */}
              <div className="border-t border-gray-700 mt-8 pt-8">
                <div className="flex flex-col md:flex-row justify-between items-center">
                  <div className="flex space-x-4 mb-4 md:mb-0">
                    <Link
                      href="https://facebook.com/vazofoundation"
                      className="text-gray-300 hover:text-white transition-colors"
                    >
                      <Facebook className="h-5 w-5" />
                      <span className="sr-only">Facebook</span>
                    </Link>
                    <Link
                      href="https://instagram.com/vazofoundation"
                      className="text-gray-300 hover:text-white transition-colors"
                    >
                      <Instagram className="h-5 w-5" />
                      <span className="sr-only">Instagram</span>
                    </Link>
                    <Link
                      href="https://linkedin.com/company/vazo-foundation"
                      className="text-gray-300 hover:text-white transition-colors"
                    >
                      <Linkedin className="h-5 w-5" />
                      <span className="sr-only">LinkedIn</span>
                    </Link>
                  </div>
                  <p className="text-gray-300 text-sm">© 2025 VAZO Foundation NPC. All rights reserved.</p>
                </div>
              </div>
            </div>
          </footer>
        </div>
      </body>
    </html>
  )
}
