"use client"

import type React from "react"

import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox"
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group"
import { Heart, Users, Phone, Home, GraduationCap, Shield } from "lucide-react"
import { useState } from "react"
import Link from "next/link"

export default function ParentSupportPage() {
  const [showForm, setShowForm] = useState(false)
  const [formData, setFormData] = useState({
    parentName: "",
    contactPhone: "",
    contactEmail: "",
    preferredContact: "",
    familySize: "",
    childrenAges: "",
    primaryChallenges: [] as string[],
    supportNeeded: [] as string[],
    urgencyLevel: "",
    previousSupport: "",
    availableTimes: [] as string[],
    languagePreference: "",
    transportNeeds: "",
    additionalInfo: "",
    consentToContact: false,
    consentToVisit: false,
  })
  const [submitted, setSubmitted] = useState(false)

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { id, value } = e.target
    setFormData((prev) => ({ ...prev, [id]: value }))
  }

  const handleSelectChange = (id: string, value: string) => {
    setFormData((prev) => ({ ...prev, [id]: value }))
  }

  const handleCheckboxChange = (id: string, checked: boolean, value?: string) => {
    if (value) {
      setFormData((prev) => {
        const currentArray = prev[id as keyof typeof prev] as string[]
        const newArray = checked ? [...currentArray, value] : currentArray.filter((item) => item !== value)
        return { ...prev, [id]: newArray }
      })
    } else {
      setFormData((prev) => ({ ...prev, [id]: checked }))
    }
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    console.log("Parent support request submitted:", formData)
    setSubmitted(true)
  }

  if (submitted) {
    return (
      <div className="container mx-auto px-4 py-12 md:py-24">
        <Card className="w-full max-w-2xl mx-auto text-center">
          <CardHeader>
            <div className="flex justify-center mb-4">
              <Heart className="h-16 w-16 text-green-500" />
            </div>
            <CardTitle className="text-3xl font-bold text-green-700">Support Request Submitted</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <p className="text-lg text-gray-700">
              Thank you for reaching out. Your request for parental support has been received and will be reviewed by
              our family support team.
            </p>
            <div className="bg-green-50 p-4 rounded-lg">
              <h3 className="font-semibold text-green-800 mb-2">
                Reference Number: PS-2024-
                {Math.floor(Math.random() * 10000)
                  .toString()
                  .padStart(4, "0")}
              </h3>
              <p className="text-sm text-green-700">
                Please save this reference number. Our family support coordinator will contact you within 48 hours.
              </p>
            </div>
            <div className="space-y-2">
              <h4 className="font-semibold text-gray-800">What happens next:</h4>
              <ul className="text-sm text-gray-600 space-y-1 text-left">
                <li>• Our family support coordinator will contact you within 48 hours</li>
                <li>• We'll schedule an initial consultation (phone or in-person)</li>
                <li>• Together we'll create a personalized support plan</li>
                <li>• You'll be matched with appropriate support services and resources</li>
                <li>• Regular check-ins will ensure ongoing support</li>
              </ul>
            </div>
            <div className="bg-blue-50 p-4 rounded-lg">
              <h4 className="font-semibold text-blue-800 mb-2">Immediate Support Available</h4>
              <p className="text-sm text-blue-700 mb-2">
                If you need immediate assistance while waiting for your coordinator:
              </p>
              <p className="text-sm text-blue-700">
                Crisis Support:{" "}
                <a href="tel:+27123456789" className="font-bold hover:underline">
                  +27 12 345 6789
                </a>
              </p>
            </div>
            <Button asChild className="bg-vazoOrange hover:bg-vazoOrange/90 text-white">
              <Link href="/">Return to Homepage</Link>
            </Button>
          </CardContent>
        </Card>
      </div>
    )
  }

  return (
    <div className="container mx-auto px-4 py-12 md:py-24">
      <div className="text-center mb-8">
        <h1 className="text-4xl font-bold mb-6 text-vazoOrange">Parent & Caregiver Support</h1>
        <p className="text-lg text-gray-700 max-w-3xl mx-auto">
          Parenting can be challenging, and you don't have to do it alone. Our family support services provide practical
          help, emotional support, and resources to strengthen families and create safe, nurturing environments for
          children.
        </p>
      </div>

      {!showForm ? (
        <>
          {/* Support Services Overview */}
          <section className="mb-12">
            <h2 className="text-3xl font-bold text-center mb-8 text-gray-800">Our Parent Support Services</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              <Card className="p-6 text-center">
                <Users className="h-12 w-12 text-vazoOrange mb-4 mx-auto" />
                <CardTitle className="text-xl font-bold mb-3">Family Coaching</CardTitle>
                <CardContent className="p-0">
                  <p className="text-gray-700">
                    One-on-one support to help families navigate challenges, improve communication, and build stronger
                    relationships.
                  </p>
                </CardContent>
              </Card>

              <Card className="p-6 text-center">
                <GraduationCap className="h-12 w-12 text-vazoOrange mb-4 mx-auto" />
                <CardTitle className="text-xl font-bold mb-3">Parenting Workshops</CardTitle>
                <CardContent className="p-0">
                  <p className="text-gray-700">
                    Educational workshops covering child development, positive discipline, and creating supportive home
                    environments.
                  </p>
                </CardContent>
              </Card>

              <Card className="p-6 text-center">
                <Home className="h-12 w-12 text-vazoOrange mb-4 mx-auto" />
                <CardTitle className="text-xl font-bold mb-3">Home Visits</CardTitle>
                <CardContent className="p-0">
                  <p className="text-gray-700">
                    In-home support services to provide practical assistance and guidance in your family's natural
                    environment.
                  </p>
                </CardContent>
              </Card>

              <Card className="p-6 text-center">
                <Phone className="h-12 w-12 text-vazoOrange mb-4 mx-auto" />
                <CardTitle className="text-xl font-bold mb-3">Crisis Support</CardTitle>
                <CardContent className="p-0">
                  <p className="text-gray-700">
                    24/7 crisis intervention and emergency support for families experiencing immediate challenges or
                    stress.
                  </p>
                </CardContent>
              </Card>

              <Card className="p-6 text-center">
                <Heart className="h-12 w-12 text-vazoOrange mb-4 mx-auto" />
                <CardTitle className="text-xl font-bold mb-3">Peer Support Groups</CardTitle>
                <CardContent className="p-0">
                  <p className="text-gray-700">
                    Connect with other parents facing similar challenges in supportive group settings facilitated by
                    trained professionals.
                  </p>
                </CardContent>
              </Card>

              <Card className="p-6 text-center">
                <Shield className="h-12 w-12 text-vazoOrange mb-4 mx-auto" />
                <CardTitle className="text-xl font-bold mb-3">Resource Connection</CardTitle>
                <CardContent className="p-0">
                  <p className="text-gray-700">
                    Help accessing community resources, government services, healthcare, education, and other essential
                    support systems.
                  </p>
                </CardContent>
              </Card>
            </div>
          </section>

          {/* Who Can Access Support */}
          <section className="mb-12 bg-green-50 p-8 rounded-lg">
            <h3 className="text-2xl font-bold text-center mb-6 text-green-800">Who Can Access Our Support?</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
              <div>
                <h4 className="text-xl font-bold mb-4 text-green-700">All Families Welcome:</h4>
                <ul className="space-y-2 text-green-600">
                  <li>• Parents and primary caregivers</li>
                  <li>• Single parents</li>
                  <li>• Grandparents raising grandchildren</li>
                  <li>• Foster and kinship caregivers</li>
                  <li>• Families with children of all ages</li>
                  <li>• Families facing any challenges</li>
                </ul>
              </div>
              <div>
                <h4 className="text-xl font-bold mb-4 text-green-700">Common Challenges We Help With:</h4>
                <ul className="space-y-2 text-green-600">
                  <li>• Child behavior and discipline</li>
                  <li>• Family communication issues</li>
                  <li>• Stress and overwhelm</li>
                  <li>• Financial pressures</li>
                  <li>• Relationship difficulties</li>
                  <li>• Accessing community resources</li>
                </ul>
              </div>
            </div>
          </section>

          {/* Emergency Support */}
          <section className="mb-12 bg-red-50 border-l-4 border-red-500 p-6 rounded-lg">
            <h3 className="text-xl font-bold text-red-700 mb-4">Need Immediate Help?</h3>
            <p className="text-red-600 mb-4">
              If you're experiencing a family crisis or need immediate support, don't wait - contact us now:
            </p>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div className="text-center">
                <p className="font-bold text-red-700">Crisis Hotline</p>
                <a href="tel:+27123456789" className="text-xl font-bold text-red-600 hover:underline">
                  +27 12 345 6789
                </a>
              </div>
              <div className="text-center">
                <p className="font-bold text-red-700">WhatsApp Support</p>
                <a href="https://wa.me/27123456789" className="text-xl font-bold text-red-600 hover:underline">
                  +27 12 345 6789
                </a>
              </div>
              <div className="text-center">
                <p className="font-bold text-red-700">Email</p>
                <a href="mailto:support@vazofoundations.org" className="text-sm font-bold text-red-600 hover:underline">
                  support@vazofoundations.org
                </a>
              </div>
            </div>
          </section>

          {/* Call to Action */}
          <section className="text-center">
            <h2 className="text-3xl font-bold mb-6 text-gray-800">Ready to Get Support?</h2>
            <p className="text-lg text-gray-700 mb-8 max-w-3xl mx-auto">
              Take the first step towards strengthening your family. Our support is confidential, non-judgmental, and
              tailored to your family's unique needs.
            </p>
            <Button
              onClick={() => setShowForm(true)}
              size="lg"
              className="bg-vazoOrange hover:bg-vazoOrange/90 text-white"
            >
              Request Parent Support
            </Button>
            <p className="text-sm text-gray-500 mt-4">
              All support services are provided free of charge and with complete confidentiality.
            </p>
          </section>
        </>
      ) : (
        /* Support Request Form */
        <Card className="w-full max-w-4xl mx-auto">
          <CardHeader>
            <CardTitle className="text-2xl font-bold text-vazoOrange flex items-center gap-2">
              <Heart className="h-6 w-6" />
              Parent Support Request Form
            </CardTitle>
            <p className="text-gray-600">
              This information helps us understand your family's needs and match you with the most appropriate support
              services.
            </p>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-8">
              {/* Contact Information */}
              <div className="bg-blue-50 p-6 rounded-lg">
                <h3 className="text-lg font-bold text-blue-800 mb-4">Contact Information</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="parentName" className="text-lg font-semibold mb-2 block">
                      Your Name *
                    </Label>
                    <Input
                      type="text"
                      id="parentName"
                      placeholder="Full name"
                      value={formData.parentName}
                      onChange={handleInputChange}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="contactPhone" className="text-lg font-semibold mb-2 block">
                      Phone Number *
                    </Label>
                    <Input
                      type="tel"
                      id="contactPhone"
                      placeholder="+27 12 345 6789"
                      value={formData.contactPhone}
                      onChange={handleInputChange}
                      required
                    />
                  </div>
                </div>
                <div className="mt-4">
                  <Label htmlFor="contactEmail" className="text-lg font-semibold mb-2 block">
                    Email Address (Optional)
                  </Label>
                  <Input
                    type="email"
                    id="contactEmail"
                    placeholder="your.email@example.com"
                    value={formData.contactEmail}
                    onChange={handleInputChange}
                  />
                </div>
                <div className="mt-4">
                  <Label htmlFor="preferredContact" className="text-lg font-semibold mb-2 block">
                    Preferred Contact Method *
                  </Label>
                  <Select onValueChange={(value) => handleSelectChange("preferredContact", value)} required>
                    <SelectTrigger>
                      <SelectValue placeholder="How would you like us to contact you?" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="phone">Phone Call</SelectItem>
                      <SelectItem value="whatsapp">WhatsApp</SelectItem>
                      <SelectItem value="sms">SMS/Text</SelectItem>
                      <SelectItem value="email">Email</SelectItem>
                      <SelectItem value="in-person">In-Person Visit</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              {/* Family Information */}
              <div>
                <h3 className="text-lg font-bold text-gray-800 mb-4">Family Information</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="familySize" className="text-lg font-semibold mb-2 block">
                      Number of People in Household *
                    </Label>
                    <Select onValueChange={(value) => handleSelectChange("familySize", value)} required>
                      <SelectTrigger>
                        <SelectValue placeholder="Select household size" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="2">2 people</SelectItem>
                        <SelectItem value="3">3 people</SelectItem>
                        <SelectItem value="4">4 people</SelectItem>
                        <SelectItem value="5">5 people</SelectItem>
                        <SelectItem value="6+">6+ people</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label htmlFor="childrenAges" className="text-lg font-semibold mb-2 block">
                      Ages of Children *
                    </Label>
                    <Input
                      type="text"
                      id="childrenAges"
                      placeholder="e.g., 3, 7, 12 years old"
                      value={formData.childrenAges}
                      onChange={handleInputChange}
                      required
                    />
                  </div>
                </div>
              </div>

              {/* Primary Challenges */}
              <div>
                <Label htmlFor="primaryChallenges" className="text-lg font-semibold mb-2 block">
                  What challenges is your family currently facing? *
                </Label>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                  {[
                    "Child behavior issues",
                    "Communication problems",
                    "Financial stress",
                    "Relationship difficulties",
                    "School-related concerns",
                    "Health issues",
                    "Housing problems",
                    "Work-life balance",
                    "Substance abuse",
                    "Mental health concerns",
                    "Domestic violence",
                    "Other family stress",
                  ].map((challenge) => (
                    <div key={challenge} className="flex items-center space-x-2">
                      <Checkbox
                        id={`challenge-${challenge}`}
                        checked={formData.primaryChallenges.includes(challenge)}
                        onCheckedChange={(checked) =>
                          handleCheckboxChange("primaryChallenges", checked as boolean, challenge)
                        }
                      />
                      <Label htmlFor={`challenge-${challenge}`} className="text-sm">
                        {challenge}
                      </Label>
                    </div>
                  ))}
                </div>
              </div>

              {/* Support Needed */}
              <div>
                <Label htmlFor="supportNeeded" className="text-lg font-semibold mb-2 block">
                  What type of support would be most helpful? *
                </Label>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                  {[
                    "One-on-one family coaching",
                    "Parenting workshops",
                    "Home visits",
                    "Peer support groups",
                    "Crisis intervention",
                    "Resource connections",
                    "Child care assistance",
                    "Financial guidance",
                    "Legal support",
                    "Mental health referrals",
                    "Educational support",
                    "Emergency assistance",
                  ].map((support) => (
                    <div key={support} className="flex items-center space-x-2">
                      <Checkbox
                        id={`support-${support}`}
                        checked={formData.supportNeeded.includes(support)}
                        onCheckedChange={(checked) =>
                          handleCheckboxChange("supportNeeded", checked as boolean, support)
                        }
                      />
                      <Label htmlFor={`support-${support}`} className="text-sm">
                        {support}
                      </Label>
                    </div>
                  ))}
                </div>
              </div>

              {/* Urgency Level */}
              <div>
                <Label htmlFor="urgencyLevel" className="text-lg font-semibold mb-2 block">
                  How urgent is your need for support? *
                </Label>
                <RadioGroup
                  value={formData.urgencyLevel}
                  onValueChange={(value) => handleSelectChange("urgencyLevel", value)}
                  className="space-y-3"
                >
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem id="crisis" value="crisis" />
                    <Label htmlFor="crisis">Crisis - Need immediate help (within 24 hours)</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem id="urgent" value="urgent" />
                    <Label htmlFor="urgent">Urgent - Need help within a week</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem id="moderate" value="moderate" />
                    <Label htmlFor="moderate">Moderate - Need help within a month</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <RadioGroupItem id="routine" value="routine" />
                    <Label htmlFor="routine">Routine - Looking for ongoing support</Label>
                  </div>
                </RadioGroup>
              </div>

              {/* Previous Support */}
              <div>
                <Label htmlFor="previousSupport" className="text-lg font-semibold mb-2 block">
                  Have you received family support services before?
                </Label>
                <Select onValueChange={(value) => handleSelectChange("previousSupport", value)}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select your experience with support services" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="no">No, this is my first time seeking support</SelectItem>
                    <SelectItem value="yes-helpful">Yes, and it was helpful</SelectItem>
                    <SelectItem value="yes-mixed">Yes, with mixed results</SelectItem>
                    <SelectItem value="yes-unhelpful">Yes, but it wasn't helpful</SelectItem>
                    <SelectItem value="unsure">I'm not sure</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {/* Availability */}
              <div>
                <Label htmlFor="availableTimes" className="text-lg font-semibold mb-2 block">
                  When are you typically available for support sessions?
                </Label>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-2">
                  {[
                    "Weekday mornings",
                    "Weekday afternoons",
                    "Weekday evenings",
                    "Weekend mornings",
                    "Weekend afternoons",
                    "Weekend evenings",
                    "Flexible",
                    "Emergency only",
                  ].map((time) => (
                    <div key={time} className="flex items-center space-x-2">
                      <Checkbox
                        id={`time-${time}`}
                        checked={formData.availableTimes.includes(time)}
                        onCheckedChange={(checked) => handleCheckboxChange("availableTimes", checked as boolean, time)}
                      />
                      <Label htmlFor={`time-${time}`} className="text-sm">
                        {time}
                      </Label>
                    </div>
                  ))}
                </div>
              </div>

              {/* Language and Transport */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="languagePreference" className="text-lg font-semibold mb-2 block">
                    Language Preference
                  </Label>
                  <Select onValueChange={(value) => handleSelectChange("languagePreference", value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Preferred language for support" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="english">English</SelectItem>
                      <SelectItem value="afrikaans">Afrikaans</SelectItem>
                      <SelectItem value="xhosa">isiXhosa</SelectItem>
                      <SelectItem value="zulu">isiZulu</SelectItem>
                      <SelectItem value="other">Other</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="transportNeeds" className="text-lg font-semibold mb-2 block">
                    Do you need help with transportation?
                  </Label>
                  <Select onValueChange={(value) => handleSelectChange("transportNeeds", value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Transportation assistance needed?" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="no">No, I have reliable transport</SelectItem>
                      <SelectItem value="sometimes">Sometimes need assistance</SelectItem>
                      <SelectItem value="yes">Yes, I need transport assistance</SelectItem>
                      <SelectItem value="prefer-home">Prefer home visits</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              {/* Additional Information */}
              <div>
                <Label htmlFor="additionalInfo" className="text-lg font-semibold mb-2 block">
                  Additional Information
                </Label>
                <Textarea
                  id="additionalInfo"
                  placeholder="Is there anything else you'd like us to know about your family's situation or needs?"
                  rows={4}
                  value={formData.additionalInfo}
                  onChange={handleInputChange}
                />
              </div>

              {/* Consent */}
              <div className="bg-gray-50 p-6 rounded-lg space-y-4">
                <h3 className="text-lg font-bold text-gray-800">Consent & Privacy</h3>
                <div className="space-y-3">
                  <div className="flex items-start space-x-2">
                    <Checkbox
                      id="consentToContact"
                      checked={formData.consentToContact}
                      onCheckedChange={(checked) => handleCheckboxChange("consentToContact", checked as boolean)}
                      className="mt-1"
                    />
                    <Label htmlFor="consentToContact" className="text-sm">
                      I consent to being contacted by Vazo Foundation's family support team regarding this request. I
                      understand that all information will be kept confidential and used only for providing support
                      services.
                    </Label>
                  </div>
                  <div className="flex items-start space-x-2">
                    <Checkbox
                      id="consentToVisit"
                      checked={formData.consentToVisit}
                      onCheckedChange={(checked) => handleCheckboxChange("consentToVisit", checked as boolean)}
                      className="mt-1"
                    />
                    <Label htmlFor="consentToVisit" className="text-sm">
                      I consent to home visits if recommended as part of my family's support plan. I understand that
                      home visits are optional and I can decline at any time.
                    </Label>
                  </div>
                </div>
                <p className="text-xs text-gray-600">
                  Your privacy is protected under POPIA. Information will only be shared with authorized support staff
                  and partner organizations as necessary to provide services.
                </p>
              </div>

              {/* Submit Button */}
              <div className="text-center space-y-4">
                <Button
                  type="submit"
                  size="lg"
                  className="bg-vazoOrange hover:bg-vazoOrange/90 text-white px-8"
                  disabled={!formData.consentToContact}
                >
                  Submit Support Request
                </Button>
                <div className="flex justify-center gap-4">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setShowForm(false)}
                    className="border-gray-300 text-gray-600"
                  >
                    Back to Information
                  </Button>
                </div>
                <p className="text-sm text-gray-500">
                  By submitting this form, you acknowledge that the information provided is accurate and you consent to
                  receiving support services from Vazo Foundation.
                </p>
              </div>
            </form>
          </CardContent>
        </Card>
      )}
    </div>
  )
}
