"use client"

import type React from "react"

export const dynamic = "force-dynamic"

import { useState, useTransition } from "react"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Badge } from "@/components/ui/badge"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Briefcase, GraduationCap, Users, TrendingUp, CheckCircle2, Loader2 } from "lucide-react"
import Image from "next/image"
import { subscribeToProject } from "@/app/actions/subscribe"

export default function VazoLinkPage() {
  const [email, setEmail] = useState("")
  const [state, setState] = useState<{ success?: boolean; message?: string } | null>(null)
  const [isPending, startTransition] = useTransition()

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault()
    setState(null)

    startTransition(async () => {
      const formData = new FormData()
      formData.append("email", email)
      formData.append("projectType", "vazo_link")

      const result = await subscribeToProject(formData)
      setState(result)

      if (result.success) {
        setEmail("")
      }
    })
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-green-50 to-blue-100">
      <div className="container mx-auto px-4 py-8">
        {/* Hero Section */}
        <div className="text-center mb-12">
          <Badge className="mb-4 bg-green-600 text-white hover:bg-green-700">Youth Opportunity Platform</Badge>
          <h1 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">VAZO Link</h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Connecting young people with education, training, and employment opportunities to unlock their full
            potential and build successful careers.
          </p>
        </div>

        {/* Main Image */}
        <div className="mb-12 rounded-xl overflow-hidden shadow-2xl">
          <Image
            src="/youth-using-opportunity-matching-app.jpg"
            alt="VAZO Link Opportunity Platform"
            width={1200}
            height={600}
            className="w-full h-auto"
            priority
          />
        </div>

        {/* Overview */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Briefcase className="h-6 w-6 text-green-600" />
              Platform Overview
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <p className="text-gray-600 leading-relaxed">
              VAZO Link is our comprehensive opportunity matching platform that connects young people aged 16-35 with
              education programs, skills training, internships, and job opportunities. Through intelligent matching and
              personalized recommendations, we help youth discover and access pathways to success.
            </p>
            <div className="grid md:grid-cols-3 gap-4 mt-6">
              <div className="flex items-start gap-3">
                <div className="bg-green-100 p-2 rounded-lg">
                  <GraduationCap className="h-5 w-5 text-green-600" />
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900">Education & Training</h3>
                  <p className="text-sm text-gray-600">Access to courses, workshops, and certifications</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <div className="bg-blue-100 p-2 rounded-lg">
                  <Briefcase className="h-5 w-5 text-blue-600" />
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900">Job Opportunities</h3>
                  <p className="text-sm text-gray-600">Connect with employers and internships</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <div className="bg-purple-100 p-2 rounded-lg">
                  <Users className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900">Mentorship Network</h3>
                  <p className="text-sm text-gray-600">Connect with professionals and peers</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Key Features */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle>Key Features</CardTitle>
            <CardDescription>Everything you need to advance your career</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-2 gap-6">
              <div className="space-y-3">
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Smart Opportunity Matching</h3>
                    <p className="text-sm text-gray-600">
                      AI-powered recommendations based on your skills, interests, and career goals
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Comprehensive Profile</h3>
                    <p className="text-sm text-gray-600">
                      Build your digital portfolio with skills, education, and work experience
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Learning Pathways</h3>
                    <p className="text-sm text-gray-600">
                      Discover structured learning paths for your desired career field
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Application Tracking</h3>
                    <p className="text-sm text-gray-600">
                      Manage all your applications and track your progress in one place
                    </p>
                  </div>
                </div>
              </div>
              <div className="space-y-3">
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Career Resources</h3>
                    <p className="text-sm text-gray-600">
                      Access guides, tips, and tools for CV writing, interviews, and career planning
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Employer Network</h3>
                    <p className="text-sm text-gray-600">
                      Connect with companies actively hiring and offering training programs
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Skills Assessment</h3>
                    <p className="text-sm text-gray-600">
                      Identify your strengths and areas for development with skill tests
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Progress Dashboard</h3>
                    <p className="text-sm text-gray-600">
                      Visualize your career journey and celebrate your achievements
                    </p>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* How It Works */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle>How It Works</CardTitle>
            <CardDescription>Your journey to opportunity in four simple steps</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-6">
              <div className="flex items-start gap-4">
                <div className="bg-green-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  1
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Create Your Profile</h3>
                  <p className="text-gray-600">
                    Sign up and build your comprehensive profile with your education, skills, interests, and career
                    goals. Complete skills assessments to help us understand your strengths.
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <div className="bg-green-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  2
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Discover Opportunities</h3>
                  <p className="text-gray-600">
                    Browse personalized recommendations for education programs, training courses, internships, and job
                    openings that match your profile and aspirations.
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <div className="bg-green-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  3
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Apply & Connect</h3>
                  <p className="text-gray-600">
                    Apply directly through the platform with your profile. Connect with mentors, join peer groups, and
                    engage with potential employers or training providers.
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <div className="bg-green-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  4
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Grow & Succeed</h3>
                  <p className="text-gray-600">
                    Track your progress, complete training programs, build your experience, and advance your career.
                    Celebrate milestones and unlock new opportunities as you grow.
                  </p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Stats */}
        <div className="grid md:grid-cols-4 gap-6 mb-8">
          <Card className="text-center">
            <CardContent className="pt-6">
              <div className="text-4xl font-bold text-green-600 mb-2">500+</div>
              <p className="text-gray-600">Active Opportunities</p>
            </CardContent>
          </Card>
          <Card className="text-center">
            <CardContent className="pt-6">
              <div className="text-4xl font-bold text-blue-600 mb-2">200+</div>
              <p className="text-gray-600">Partner Organizations</p>
            </CardContent>
          </Card>
          <Card className="text-center">
            <CardContent className="pt-6">
              <div className="text-4xl font-bold text-purple-600 mb-2">85%</div>
              <p className="text-gray-600">Placement Success Rate</p>
            </CardContent>
          </Card>
          <Card className="text-center">
            <CardContent className="pt-6">
              <div className="text-4xl font-bold text-orange-600 mb-2">1000+</div>
              <p className="text-gray-600">Youth Empowered</p>
            </CardContent>
          </Card>
        </div>

        {/* Join Waitlist */}
        <Card className="mb-8 bg-gradient-to-br from-green-600 to-blue-700 text-white">
          <CardHeader>
            <CardTitle className="text-white">Join the Waitlist</CardTitle>
            <CardDescription className="text-green-100">
              Be among the first to access VAZO Link when we launch
            </CardDescription>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="flex gap-4">
                <Input
                  type="email"
                  placeholder="Enter your email"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  required
                  className="bg-white/10 border-white/20 text-white placeholder:text-white/60"
                  disabled={isPending}
                />
                <Button type="submit" variant="secondary" disabled={isPending}>
                  {isPending ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Joining...
                    </>
                  ) : (
                    "Join Waitlist"
                  )}
                </Button>
              </div>
              {state?.message && (
                <Alert variant={state.success ? "default" : "destructive"}>
                  <AlertDescription>{state.message}</AlertDescription>
                </Alert>
              )}
            </form>
          </CardContent>
        </Card>

        {/* Success Stories */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="h-6 w-6 text-green-600" />
              Success Stories
            </CardTitle>
            <CardDescription>Real impact on young people's lives</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-3 gap-6">
              <div className="space-y-3">
                <div className="aspect-video bg-gradient-to-br from-green-400 to-blue-500 rounded-lg"></div>
                <h3 className="font-semibold text-gray-900">From Unemployed to IT Professional</h3>
                <p className="text-sm text-gray-600">
                  "VAZO Link helped me discover a coding bootcamp that changed my life. Now I'm working at a tech
                  company!"
                </p>
                <p className="text-sm font-semibold text-gray-900">- Thabo M., Cape Town</p>
              </div>
              <div className="space-y-3">
                <div className="aspect-video bg-gradient-to-br from-purple-400 to-pink-500 rounded-lg"></div>
                <h3 className="font-semibold text-gray-900">Built My Own Business</h3>
                <p className="text-sm text-gray-600">
                  "The entrepreneurship training I found through VAZO Link gave me the skills and confidence to start my
                  own business."
                </p>
                <p className="text-sm font-semibold text-gray-900">- Lerato K., Johannesburg</p>
              </div>
              <div className="space-y-3">
                <div className="aspect-video bg-gradient-to-br from-orange-400 to-red-500 rounded-lg"></div>
                <h3 className="font-semibold text-gray-900">Found My Calling in Healthcare</h3>
                <p className="text-sm text-gray-600">
                  "I connected with a nursing program through the platform and now I'm making a difference in my
                  community every day."
                </p>
                <p className="text-sm font-semibold text-gray-900">- Nomsa D., Durban</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  )
}
