"use client"

import type React from "react"

export const dynamic = "force-dynamic"

import { useState, useTransition } from "react"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Badge } from "@/components/ui/badge"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Shield, Users, Bell, MapPin, Phone, AlertTriangle, CheckCircle2, Loader2 } from "lucide-react"
import Image from "next/image"
import { subscribeToProject } from "@/app/actions/subscribe"

export default function VigilantPage() {
  const [email, setEmail] = useState("")
  const [state, setState] = useState<{ success?: boolean; message?: string } | null>(null)
  const [isPending, startTransition] = useTransition()

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault()
    setState(null)

    startTransition(async () => {
      const formData = new FormData()
      formData.append("email", email)
      formData.append("projectType", "vigilant")

      const result = await subscribeToProject(formData)
      setState(result)

      if (result.success) {
        setEmail("")
      }
    })
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 to-indigo-100">
      <div className="container mx-auto px-4 py-8">
        {/* Hero Section */}
        <div className="text-center mb-12">
          <Badge className="mb-4 bg-blue-600 text-white hover:bg-blue-700">Community Safety Platform</Badge>
          <h1 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Vigilant</h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Real-time community safety network empowering residents to report incidents, coordinate responses, and build
            safer neighborhoods together.
          </p>
        </div>

        {/* Main Image */}
        <div className="mb-12 rounded-xl overflow-hidden shadow-2xl">
          <Image
            src="/community-safety-app-interface.jpg"
            alt="Vigilant Community Safety Platform Interface"
            width={1200}
            height={600}
            className="w-full h-auto"
            priority
          />
        </div>

        {/* Overview */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Shield className="h-6 w-6 text-blue-600" />
              Platform Overview
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <p className="text-gray-600 leading-relaxed">
              Vigilant is our innovative community safety platform that connects neighbors, local organizations, and
              emergency responders in real-time. Through our mobile app and web platform, community members can report
              incidents, share safety updates, and coordinate rapid responses to keep their neighborhoods safe.
            </p>
            <div className="grid md:grid-cols-3 gap-4 mt-6">
              <div className="flex items-start gap-3">
                <div className="bg-blue-100 p-2 rounded-lg">
                  <Bell className="h-5 w-5 text-blue-600" />
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900">Instant Alerts</h3>
                  <p className="text-sm text-gray-600">Real-time notifications for safety incidents</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <div className="bg-green-100 p-2 rounded-lg">
                  <MapPin className="h-5 w-5 text-green-600" />
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900">Location Tracking</h3>
                  <p className="text-sm text-gray-600">Map-based incident reporting and monitoring</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <div className="bg-purple-100 p-2 rounded-lg">
                  <Users className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900">Community Network</h3>
                  <p className="text-sm text-gray-600">Connect with neighbors and responders</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Key Features */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle>Key Features</CardTitle>
            <CardDescription>Comprehensive tools for community safety</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-2 gap-6">
              <div className="space-y-3">
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Incident Reporting</h3>
                    <p className="text-sm text-gray-600">
                      Quickly report safety concerns with photos, location, and detailed descriptions
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Emergency Contacts</h3>
                    <p className="text-sm text-gray-600">
                      Direct access to local emergency services and community response teams
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Safety Map</h3>
                    <p className="text-sm text-gray-600">
                      Interactive map showing incident locations and high-activity areas
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Neighborhood Watch</h3>
                    <p className="text-sm text-gray-600">Coordinate with local watch groups and volunteer patrols</p>
                  </div>
                </div>
              </div>
              <div className="space-y-3">
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Real-Time Updates</h3>
                    <p className="text-sm text-gray-600">Receive instant notifications about incidents in your area</p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Anonymous Reporting</h3>
                    <p className="text-sm text-gray-600">
                      Option to report sensitive incidents while maintaining privacy
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Safety Resources</h3>
                    <p className="text-sm text-gray-600">
                      Access guides, tips, and educational materials for community safety
                    </p>
                  </div>
                </div>
                <div className="flex items-start gap-3">
                  <CheckCircle2 className="h-5 w-5 text-green-600 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Data Analytics</h3>
                    <p className="text-sm text-gray-600">Track trends and patterns to improve safety strategies</p>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* How It Works */}
        <Card className="mb-8">
          <CardHeader>
            <CardTitle>How It Works</CardTitle>
            <CardDescription>Simple steps to a safer community</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-6">
              <div className="flex items-start gap-4">
                <div className="bg-blue-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  1
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Download & Register</h3>
                  <p className="text-gray-600">
                    Download the Vigilant app and create your account with basic information. Verify your address to
                    join your neighborhood network.
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <div className="bg-blue-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  2
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Set Your Preferences</h3>
                  <p className="text-gray-600">
                    Choose what types of alerts you want to receive and customize your notification settings based on
                    distance and incident type.
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <div className="bg-blue-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  3
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Report & Monitor</h3>
                  <p className="text-gray-600">
                    Report incidents as they happen and monitor your neighborhood's safety feed. Engage with other
                    community members and coordinate responses.
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <div className="bg-blue-600 text-white rounded-full w-8 h-8 flex items-center justify-center flex-shrink-0 font-bold">
                  4
                </div>
                <div>
                  <h3 className="font-semibold text-gray-900 mb-1">Build Community</h3>
                  <p className="text-gray-600">
                    Connect with neighbors, join watch groups, and participate in community safety initiatives. Work
                    together to make your area safer.
                  </p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Stay Updated */}
        <Card className="mb-8 bg-gradient-to-br from-blue-600 to-indigo-700 text-white">
          <CardHeader>
            <CardTitle className="text-white">Stay Updated on Launch</CardTitle>
            <CardDescription className="text-blue-100">
              Be the first to know when Vigilant launches in your area
            </CardDescription>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="flex gap-4">
                <Input
                  type="email"
                  placeholder="Enter your email"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  required
                  className="bg-white/10 border-white/20 text-white placeholder:text-white/60"
                  disabled={isPending}
                />
                <Button type="submit" variant="secondary" disabled={isPending}>
                  {isPending ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Subscribing...
                    </>
                  ) : (
                    "Subscribe"
                  )}
                </Button>
              </div>
              {state?.message && (
                <Alert variant={state.success ? "default" : "destructive"}>
                  <AlertDescription>{state.message}</AlertDescription>
                </Alert>
              )}
            </form>
          </CardContent>
        </Card>

        {/* Emergency Contact */}
        <Card className="border-red-200 bg-red-50">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-red-800">
              <AlertTriangle className="h-6 w-6" />
              Emergency Contact
            </CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-gray-700 mb-4">
              If you are experiencing an emergency, please contact emergency services immediately:
            </p>
            <div className="flex flex-col sm:flex-row gap-4">
              <Button variant="destructive" size="lg" className="gap-2">
                <Phone className="h-5 w-5" />
                Call 10111 (Police)
              </Button>
              <Button
                variant="outline"
                size="lg"
                className="gap-2 border-red-300 text-red-700 hover:bg-red-100 bg-transparent"
              >
                <Phone className="h-5 w-5" />
                Call 10177 (Ambulance)
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  )
}
