"use client"

import type React from "react"

import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox"
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group"
import { AlertTriangle, Shield, Phone, Lock, Eye, EyeOff, Heart } from "lucide-react"
import { useState } from "react"
import Link from "next/link"

export default function ReportIncidentPage() {
  const [formData, setFormData] = useState({
    incidentType: "",
    urgencyLevel: "",
    incidentDate: "",
    incidentTime: "",
    location: "",
    locationDetails: "",
    personsInvolved: "",
    incidentDescription: "",
    witnessInfo: "",
    previousReports: "",
    immediateAction: "",
    reporterType: "",
    contactMethod: "",
    contactDetails: "",
    anonymousReport: true,
    consentToContact: false,
    additionalSupport: [] as string[],
  })

  const [showContactDetails, setShowContactDetails] = useState(false)
  const [submitted, setSubmitted] = useState(false)

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { id, value } = e.target
    setFormData((prev) => ({ ...prev, [id]: value }))
  }

  const handleSelectChange = (id: string, value: string) => {
    setFormData((prev) => ({ ...prev, [id]: value }))
  }

  const handleCheckboxChange = (id: string, checked: boolean, value?: string) => {
    if (value) {
      setFormData((prev) => {
        const currentArray = prev[id as keyof typeof prev] as string[]
        const newArray = checked ? [...currentArray, value] : currentArray.filter((item) => item !== value)
        return { ...prev, [id]: newArray }
      })
    } else {
      setFormData((prev) => ({ ...prev, [id]: checked }))
    }
  }

  const handleAnonymousToggle = (anonymous: boolean) => {
    setFormData((prev) => ({ ...prev, anonymousReport: anonymous }))
    setShowContactDetails(!anonymous)
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    // Simulate form submission
    console.log("Incident report submitted:", formData)
    setSubmitted(true)

    // In a real implementation, this would send the data securely to the backend
    // and generate a reference number for the reporter
  }

  if (submitted) {
    return (
      <div className="container mx-auto px-4 py-12 md:py-24">
        <Card className="w-full max-w-2xl mx-auto text-center">
          <CardHeader>
            <div className="flex justify-center mb-4">
              <Shield className="h-16 w-16 text-green-500" />
            </div>
            <CardTitle className="text-3xl font-bold text-green-700">Report Submitted Successfully</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <p className="text-lg text-gray-700">
              Thank you for reporting this incident. Your report has been received and will be handled with the utmost
              confidentiality.
            </p>
            <div className="bg-green-50 p-4 rounded-lg">
              <h3 className="font-semibold text-green-800 mb-2">
                Reference Number: VF-2024-
                {Math.floor(Math.random() * 10000)
                  .toString()
                  .padStart(4, "0")}
              </h3>
              <p className="text-sm text-green-700">
                Please save this reference number for your records. You can use it to follow up on your report.
              </p>
            </div>
            <div className="space-y-2">
              <h4 className="font-semibold text-gray-800">What happens next:</h4>
              <ul className="text-sm text-gray-600 space-y-1 text-left">
                <li>• Your report will be reviewed by our trained safeguarding team</li>
                <li>• If immediate action is required, emergency services will be contacted</li>
                <li>• We will coordinate with appropriate authorities and partner organizations</li>
                <li>• Follow-up support will be provided as needed</li>
              </ul>
            </div>
            <Button asChild className="bg-vazoOrange hover:bg-vazoOrange/90 text-white">
              <Link href="/">Return to Homepage</Link>
            </Button>
          </CardContent>
        </Card>
      </div>
    )
  }

  return (
    <div className="container mx-auto px-4 py-12 md:py-24">
      <div className="text-center mb-8">
        <h1 className="text-4xl font-bold mb-6 text-vazoOrange">Anonymous Incident Reporting</h1>
        <p className="text-lg text-gray-700 max-w-3xl mx-auto">
          Report incidents safely and confidentially. Your safety and privacy are our top priorities. All reports are
          handled by trained professionals with strict confidentiality protocols.
        </p>
      </div>

      {/* Emergency Notice */}
      <div className="mb-8 bg-red-50 border-l-4 border-red-500 p-6 rounded-lg">
        <div className="flex items-center mb-4">
          <AlertTriangle className="h-8 w-8 text-red-500 mr-3" />
          <h2 className="text-2xl font-bold text-red-700">Emergency Situations</h2>
        </div>
        <p className="text-red-600 mb-4">
          If someone is in immediate danger, please contact emergency services immediately:
        </p>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="text-center">
            <p className="font-bold text-red-700">Police Emergency</p>
            <a href="tel:10111" className="text-2xl font-bold text-red-600 hover:underline">
              10111
            </a>
          </div>
          <div className="text-center">
            <p className="font-bold text-red-700">Ambulance</p>
            <a href="tel:10177" className="text-2xl font-bold text-red-600 hover:underline">
              10177
            </a>
          </div>
          <div className="text-center">
            <p className="font-bold text-red-700">Childline</p>
            <a href="tel:08000055555" className="text-2xl font-bold text-red-600 hover:underline">
              0800 005 5555
            </a>
          </div>
        </div>
      </div>

      {/* Reporting Form */}
      <Card className="w-full max-w-4xl mx-auto">
        <CardHeader>
          <CardTitle className="text-2xl font-bold text-vazoOrange flex items-center gap-2">
            <Lock className="h-6 w-6" />
            Confidential Incident Report
          </CardTitle>
          <p className="text-gray-600">
            All information provided will be kept strictly confidential and used only for safeguarding purposes.
          </p>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-8">
            {/* Anonymous Reporting Option */}
            <div className="bg-blue-50 p-6 rounded-lg">
              <h3 className="text-lg font-bold text-blue-800 mb-4">Reporting Preference</h3>
              <RadioGroup
                value={formData.anonymousReport ? "anonymous" : "identified"}
                onValueChange={(value) => handleAnonymousToggle(value === "anonymous")}
                className="space-y-3"
              >
                <div className="flex items-center space-x-2">
                  <RadioGroupItem id="anonymous" value="anonymous" />
                  <Label htmlFor="anonymous" className="flex items-center gap-2">
                    <EyeOff className="h-4 w-4" />
                    Anonymous Report (Recommended for safety)
                  </Label>
                </div>
                <div className="flex items-center space-x-2">
                  <RadioGroupItem id="identified" value="identified" />
                  <Label htmlFor="identified" className="flex items-center gap-2">
                    <Eye className="h-4 w-4" />
                    Provide contact details for follow-up
                  </Label>
                </div>
              </RadioGroup>
            </div>

            {/* Incident Type */}
            <div>
              <Label htmlFor="incidentType" className="text-lg font-semibold mb-2 block">
                Type of Incident *
              </Label>
              <Select onValueChange={(value) => handleSelectChange("incidentType", value)} required>
                <SelectTrigger>
                  <SelectValue placeholder="Select the type of incident" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="child-abuse-physical">Physical Abuse of a Child</SelectItem>
                  <SelectItem value="child-abuse-sexual">Sexual Abuse of a Child</SelectItem>
                  <SelectItem value="child-abuse-emotional">Emotional/Psychological Abuse</SelectItem>
                  <SelectItem value="child-neglect">Child Neglect</SelectItem>
                  <SelectItem value="domestic-violence">Domestic Violence</SelectItem>
                  <SelectItem value="community-safety">Community Safety Concern</SelectItem>
                  <SelectItem value="substance-abuse">Substance Abuse</SelectItem>
                  <SelectItem value="bullying">Bullying or Harassment</SelectItem>
                  <SelectItem value="exploitation">Child Exploitation</SelectItem>
                  <SelectItem value="other">Other Safety Concern</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Urgency Level */}
            <div>
              <Label htmlFor="urgencyLevel" className="text-lg font-semibold mb-2 block">
                Urgency Level *
              </Label>
              <Select onValueChange={(value) => handleSelectChange("urgencyLevel", value)} required>
                <SelectTrigger>
                  <SelectValue placeholder="How urgent is this situation?" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="immediate">Immediate - Someone is in danger right now</SelectItem>
                  <SelectItem value="urgent">Urgent - Action needed within 24 hours</SelectItem>
                  <SelectItem value="important">Important - Action needed within a week</SelectItem>
                  <SelectItem value="routine">Routine - General concern to be addressed</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Date and Time */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="incidentDate" className="text-lg font-semibold mb-2 block">
                  Date of Incident
                </Label>
                <Input type="date" id="incidentDate" value={formData.incidentDate} onChange={handleInputChange} />
              </div>
              <div>
                <Label htmlFor="incidentTime" className="text-lg font-semibold mb-2 block">
                  Approximate Time
                </Label>
                <Input type="time" id="incidentTime" value={formData.incidentTime} onChange={handleInputChange} />
              </div>
            </div>

            {/* Location */}
            <div>
              <Label htmlFor="location" className="text-lg font-semibold mb-2 block">
                General Location *
              </Label>
              <Select onValueChange={(value) => handleSelectChange("location", value)} required>
                <SelectTrigger>
                  <SelectValue placeholder="Where did this incident occur?" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="home">Home/Residence</SelectItem>
                  <SelectItem value="school">School</SelectItem>
                  <SelectItem value="community-center">Community Center</SelectItem>
                  <SelectItem value="public-space">Public Space</SelectItem>
                  <SelectItem value="online">Online/Digital</SelectItem>
                  <SelectItem value="workplace">Workplace</SelectItem>
                  <SelectItem value="transport">Public Transport</SelectItem>
                  <SelectItem value="other">Other Location</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label htmlFor="locationDetails" className="text-lg font-semibold mb-2 block">
                Location Details (Optional)
              </Label>
              <Input
                type="text"
                id="locationDetails"
                placeholder="e.g., Near the park, specific street name (only if safe to share)"
                value={formData.locationDetails}
                onChange={handleInputChange}
              />
            </div>

            {/* Persons Involved */}
            <div>
              <Label htmlFor="personsInvolved" className="text-lg font-semibold mb-2 block">
                Persons Involved
              </Label>
              <Textarea
                id="personsInvolved"
                placeholder="Describe the people involved (ages, relationships, etc.) - no need to provide names if you prefer not to"
                rows={3}
                value={formData.personsInvolved}
                onChange={handleInputChange}
              />
            </div>

            {/* Incident Description */}
            <div>
              <Label htmlFor="incidentDescription" className="text-lg font-semibold mb-2 block">
                Incident Description *
              </Label>
              <Textarea
                id="incidentDescription"
                placeholder="Please describe what happened. Include as much detail as you feel comfortable sharing."
                rows={5}
                value={formData.incidentDescription}
                onChange={handleInputChange}
                required
              />
            </div>

            {/* Witness Information */}
            <div>
              <Label htmlFor="witnessInfo" className="text-lg font-semibold mb-2 block">
                Witness Information (Optional)
              </Label>
              <Textarea
                id="witnessInfo"
                placeholder="Were there any witnesses? What did they see or hear?"
                rows={3}
                value={formData.witnessInfo}
                onChange={handleInputChange}
              />
            </div>

            {/* Previous Reports */}
            <div>
              <Label htmlFor="previousReports" className="text-lg font-semibold mb-2 block">
                Previous Reports
              </Label>
              <Select onValueChange={(value) => handleSelectChange("previousReports", value)}>
                <SelectTrigger>
                  <SelectValue placeholder="Has this been reported before?" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="no">No, this is the first report</SelectItem>
                  <SelectItem value="yes-police">Yes, reported to police</SelectItem>
                  <SelectItem value="yes-social-services">Yes, reported to social services</SelectItem>
                  <SelectItem value="yes-school">Yes, reported to school</SelectItem>
                  <SelectItem value="yes-other">Yes, reported elsewhere</SelectItem>
                  <SelectItem value="unsure">I'm not sure</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Immediate Action */}
            <div>
              <Label htmlFor="immediateAction" className="text-lg font-semibold mb-2 block">
                Immediate Action Needed
              </Label>
              <Textarea
                id="immediateAction"
                placeholder="What immediate help or action do you think is needed?"
                rows={3}
                value={formData.immediateAction}
                onChange={handleInputChange}
              />
            </div>

            {/* Reporter Type */}
            <div>
              <Label htmlFor="reporterType" className="text-lg font-semibold mb-2 block">
                Your Relationship to the Situation
              </Label>
              <Select onValueChange={(value) => handleSelectChange("reporterType", value)}>
                <SelectTrigger>
                  <SelectValue placeholder="How are you connected to this incident?" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="victim">I am the person affected</SelectItem>
                  <SelectItem value="parent-guardian">Parent/Guardian</SelectItem>
                  <SelectItem value="family-member">Family Member</SelectItem>
                  <SelectItem value="friend">Friend</SelectItem>
                  <SelectItem value="neighbor">Neighbor</SelectItem>
                  <SelectItem value="teacher">Teacher/School Staff</SelectItem>
                  <SelectItem value="community-member">Community Member</SelectItem>
                  <SelectItem value="professional">Professional (social worker, healthcare, etc.)</SelectItem>
                  <SelectItem value="prefer-not-to-say">Prefer not to say</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {/* Contact Information (if not anonymous) */}
            {showContactDetails && (
              <div className="bg-yellow-50 p-6 rounded-lg space-y-4">
                <h3 className="text-lg font-bold text-yellow-800 mb-4">Contact Information</h3>
                <div>
                  <Label htmlFor="contactMethod" className="text-lg font-semibold mb-2 block">
                    Preferred Contact Method
                  </Label>
                  <Select onValueChange={(value) => handleSelectChange("contactMethod", value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="How would you like us to contact you?" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="phone">Phone Call</SelectItem>
                      <SelectItem value="sms">SMS/Text Message</SelectItem>
                      <SelectItem value="email">Email</SelectItem>
                      <SelectItem value="in-person">In-Person Meeting</SelectItem>
                      <SelectItem value="no-contact">Do not contact me</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="contactDetails" className="text-lg font-semibold mb-2 block">
                    Contact Details
                  </Label>
                  <Input
                    type="text"
                    id="contactDetails"
                    placeholder="Phone number or email address"
                    value={formData.contactDetails}
                    onChange={handleInputChange}
                  />
                </div>
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="consentToContact"
                    checked={formData.consentToContact}
                    onCheckedChange={(checked) => handleCheckboxChange("consentToContact", checked as boolean)}
                  />
                  <Label htmlFor="consentToContact" className="text-sm">
                    I consent to being contacted by Vazo Foundation regarding this report
                  </Label>
                </div>
              </div>
            )}

            {/* Additional Support */}
            <div>
              <Label htmlFor="additionalSupport" className="text-lg font-semibold mb-2 block">
                Additional Support Needed (Optional)
              </Label>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-2">
                {[
                  "Counseling/Mental Health Support",
                  "Legal Assistance",
                  "Emergency Accommodation",
                  "Medical Care",
                  "Financial Support",
                  "Educational Support",
                  "Family Mediation",
                  "Other Support Services",
                ].map((option) => (
                  <div key={option} className="flex items-center space-x-2">
                    <Checkbox
                      id={`support-${option}`}
                      checked={formData.additionalSupport.includes(option)}
                      onCheckedChange={(checked) =>
                        handleCheckboxChange("additionalSupport", checked as boolean, option)
                      }
                    />
                    <Label htmlFor={`support-${option}`} className="text-sm">
                      {option}
                    </Label>
                  </div>
                ))}
              </div>
            </div>

            {/* Privacy Notice */}
            <div className="bg-gray-50 p-6 rounded-lg">
              <h3 className="text-lg font-bold text-gray-800 mb-4">Privacy & Confidentiality</h3>
              <ul className="text-sm text-gray-600 space-y-2">
                <li>• Your report will be handled with strict confidentiality</li>
                <li>• Information will only be shared with authorized personnel on a need-to-know basis</li>
                <li>• We comply with all data protection laws (POPIA)</li>
                <li>• Anonymous reports cannot be traced back to you</li>
                <li>• We may need to share information with authorities if required by law for child protection</li>
              </ul>
            </div>

            {/* Submit Button */}
            <div className="text-center">
              <Button type="submit" size="lg" className="bg-vazoOrange hover:bg-vazoOrange/90 text-white px-8">
                Submit Confidential Report
              </Button>
              <p className="text-sm text-gray-500 mt-4">
                By submitting this report, you acknowledge that the information provided is accurate to the best of your
                knowledge.
              </p>
            </div>
          </form>
        </CardContent>
      </Card>

      {/* Support Resources */}
      <div className="mt-12 bg-blue-50 p-8 rounded-lg">
        <h3 className="text-2xl font-bold text-center mb-6 text-blue-800">Support Resources</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="text-center">
            <Phone className="h-8 w-8 text-blue-600 mx-auto mb-2" />
            <h4 className="font-bold text-blue-700 mb-2">24/7 Crisis Support</h4>
            <p className="text-sm text-blue-600">
              Lifeline:{" "}
              <a href="tel:0861322322" className="font-bold hover:underline">
                0861 322 322
              </a>
            </p>
          </div>
          <div className="text-center">
            <Shield className="h-8 w-8 text-blue-600 mx-auto mb-2" />
            <h4 className="font-bold text-blue-700 mb-2">Child Protection</h4>
            <p className="text-sm text-blue-600">
              DSD Hotline:{" "}
              <a href="tel:0800220250" className="font-bold hover:underline">
                0800 220 250
              </a>
            </p>
          </div>
          <div className="text-center">
            <Heart className="h-8 w-8 text-blue-600 mx-auto mb-2" />
            <h4 className="font-bold text-blue-700 mb-2">Our Support Team</h4>
            <p className="text-sm text-blue-600">
              Email:{" "}
              <a href="mailto:support@vazofoundations.org" className="font-bold hover:underline">
                support@vazofoundations.org
              </a>
            </p>
          </div>
        </div>
      </div>
    </div>
  )
}
