"use client"

import * as React from "react"
import Link from "next/link"
import { cn } from "@/lib/utils"
import {
  NavigationMenu,
  NavigationMenuContent,
  NavigationMenuItem,
  NavigationMenuLink,
  NavigationMenuList,
  NavigationMenuTrigger,
  navigationMenuTriggerStyle,
} from "@/components/ui/navigation-menu"

const aboutItems = [
  {
    title: "About",
    href: "/about",
    description: "Learn about our mission, vision, and values",
  },
  {
    title: "Our Story",
    href: "/about/our-story",
    description: "Discover our journey and impact in the community",
  },
  {
    title: "Meet the Directors",
    href: "/about/directors",
    description: "Get to know our dedicated leadership team",
  },
  {
    title: "Gallery",
    href: "/about/gallery",
    description: "See our programs and community impact in photos",
  },
]

const programItems = [
  {
    title: "All Programs",
    href: "/programs",
    description: "Overview of all our community programs",
  },
  {
    title: "Skills Workshops",
    href: "/programs/skills-workshops",
    description: "Practical skills training for youth development",
  },
  {
    title: "Job Readiness Coaching",
    href: "/programs/job-readiness-coaching",
    description: "Prepare for employment with professional coaching",
  },
  {
    title: "Mentorship Circles",
    href: "/programs/mentorship-circles",
    description: "Connect with mentors and peer support groups",
  },
  {
    title: "Health & Wellness",
    href: "/programs/health-wellness",
    description: "Mental health support and wellness programs",
  },
  {
    title: "Vigilant",
    href: "/programs/vigilant",
    description: "Community safety platform (in development)",
  },
  {
    title: "VAZO Link",
    href: "/programs/vazo-link",
    description: "Youth opportunity matching platform (under construction)",
  },
]

const getInvolvedItems = [
  {
    title: "Get Involved",
    href: "/get-involved",
    description: "Ways to support and participate in our mission",
  },
  {
    title: "Volunteer",
    href: "/get-involved/volunteer",
    description: "Join our team of dedicated volunteers",
  },
  {
    title: "Partner with Us",
    href: "/get-involved/partner",
    description: "Collaborate with us to create greater impact",
  },
  {
    title: "Report an Incident",
    href: "/report-incident",
    description: "Report safety concerns or incidents in your community",
  },
]

export function MainNav() {
  return (
    <NavigationMenu>
      <NavigationMenuList>
        <NavigationMenuItem>
          <Link href="/" legacyBehavior passHref>
            <NavigationMenuLink className={navigationMenuTriggerStyle()}>Home</NavigationMenuLink>
          </Link>
        </NavigationMenuItem>

        <NavigationMenuItem>
          <NavigationMenuTrigger>About</NavigationMenuTrigger>
          <NavigationMenuContent>
            <ul className="grid w-[400px] gap-3 p-4 md:w-[500px] md:grid-cols-2 lg:w-[600px]">
              {aboutItems.map((item) => (
                <ListItem key={item.title} title={item.title} href={item.href}>
                  {item.description}
                </ListItem>
              ))}
            </ul>
          </NavigationMenuContent>
        </NavigationMenuItem>

        <NavigationMenuItem>
          <NavigationMenuTrigger>Programs</NavigationMenuTrigger>
          <NavigationMenuContent>
            <ul className="grid w-[400px] gap-3 p-4 md:w-[500px] md:grid-cols-2 lg:w-[600px]">
              {programItems.map((item) => (
                <ListItem key={item.title} title={item.title} href={item.href}>
                  {item.description}
                </ListItem>
              ))}
            </ul>
          </NavigationMenuContent>
        </NavigationMenuItem>

        <NavigationMenuItem>
          <NavigationMenuTrigger>Get Involved</NavigationMenuTrigger>
          <NavigationMenuContent>
            <ul className="grid w-[400px] gap-3 p-4 md:w-[500px] md:grid-cols-2 lg:w-[600px]">
              {getInvolvedItems.map((item) => (
                <ListItem key={item.title} title={item.title} href={item.href}>
                  {item.description}
                </ListItem>
              ))}
            </ul>
          </NavigationMenuContent>
        </NavigationMenuItem>

        <NavigationMenuItem>
          <Link href="/resources" legacyBehavior passHref>
            <NavigationMenuLink className={navigationMenuTriggerStyle()}>Resources</NavigationMenuLink>
          </Link>
        </NavigationMenuItem>

        <NavigationMenuItem>
          <Link href="/parent-support" legacyBehavior passHref>
            <NavigationMenuLink className={navigationMenuTriggerStyle()}>Parent Support</NavigationMenuLink>
          </Link>
        </NavigationMenuItem>
      </NavigationMenuList>
    </NavigationMenu>
  )
}

const ListItem = React.forwardRef<React.ElementRef<"a">, React.ComponentPropsWithoutRef<"a">>(
  ({ className, title, children, ...props }, ref) => {
    return (
      <li>
        <NavigationMenuLink asChild>
          <a
            ref={ref}
            className={cn(
              "block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors hover:bg-accent hover:text-accent-foreground focus:bg-accent focus:text-accent-foreground",
              className,
            )}
            {...props}
          >
            <div className="text-sm font-medium leading-none">{title}</div>
            <p className="line-clamp-2 text-sm leading-snug text-muted-foreground">{children}</p>
          </a>
        </NavigationMenuLink>
      </li>
    )
  },
)
ListItem.displayName = "ListItem"
